import { Router } from 'express';
import { authenticateToken } from '../middleware/auth.js';
import { backupManager } from '../modules/backup/BackupManager.js';
import logger from '../utils/logger.js';
const router = Router();
router.use(authenticateToken);
router.get('/', async (req, res) => {
    try {
        const data = await backupManager.listBackups();
        res.json({ success: true, data });
    }
    catch (error) {
        logger.error('获取备份列表失败:', error);
        res.status(500).json({ success: false, message: error.message || '获取备份列表失败' });
    }
});
router.post('/create', async (req, res) => {
    try {
        const { backupName, sourcePath, maxKeep = 10 } = req.body || {};
        if (!backupName || !sourcePath) {
            return res.status(400).json({ success: false, message: '缺少必要参数: backupName 或 sourcePath' });
        }
        const result = await backupManager.createBackup(String(backupName), String(sourcePath), Number(maxKeep));
        res.json({ success: true, data: result });
    }
    catch (error) {
        logger.error('创建备份失败:', error);
        res.status(500).json({ success: false, message: error.message || '创建备份失败' });
    }
});
router.post('/restore', async (req, res) => {
    try {
        const { backupName, fileName } = req.body || {};
        if (!backupName || !fileName) {
            return res.status(400).json({ success: false, message: '缺少必要参数: backupName 或 fileName' });
        }
        const data = await backupManager.restoreBackup(String(backupName), String(fileName));
        res.json({ success: true, data, message: '恢复任务已完成' });
    }
    catch (error) {
        logger.error('恢复备份失败:', error);
        res.status(500).json({ success: false, message: error.message || '恢复备份失败' });
    }
});
router.delete('/file', async (req, res) => {
    try {
        const { backupName, fileName } = (req.query || {});
        if (!backupName || !fileName) {
            return res.status(400).json({ success: false, message: '缺少必要参数: backupName 或 fileName' });
        }
        await backupManager.deleteBackupFile(String(backupName), String(fileName));
        res.json({ success: true, message: '删除成功' });
    }
    catch (error) {
        logger.error('删除备份文件失败:', error);
        res.status(500).json({ success: false, message: error.message || '删除备份文件失败' });
    }
});
router.delete('/folder', async (req, res) => {
    try {
        const { backupName } = (req.query || {});
        if (!backupName) {
            return res.status(400).json({ success: false, message: '缺少必要参数: backupName' });
        }
        await backupManager.deleteBackupFolder(String(backupName));
        res.json({ success: true, message: '备份文件夹已删除' });
    }
    catch (error) {
        logger.error('删除备份文件夹失败:', error);
        res.status(500).json({ success: false, message: error.message || '删除备份文件夹失败' });
    }
});
router.get('/download', async (req, res) => {
    try {
        const { backupName, fileName } = (req.query || {});
        if (!backupName || !fileName) {
            return res.status(400).json({ success: false, message: '缺少必要参数: backupName 或 fileName' });
        }
        const filePath = await backupManager.getBackupFilePath(String(backupName), String(fileName));
        res.setHeader('Content-Disposition', `attachment; filename="${encodeURIComponent(fileName)}"`);
        res.setHeader('Content-Type', 'application/octet-stream');
        res.sendFile(filePath, (err) => {
            if (err) {
                logger.error('下载备份文件失败:', err);
                if (!res.headersSent) {
                    res.status(500).json({ success: false, message: '下载备份文件失败' });
                }
            }
        });
    }
    catch (error) {
        logger.error('下载备份文件失败:', error);
        if (!res.headersSent) {
            res.status(500).json({ success: false, message: error.message || '下载备份文件失败' });
        }
    }
});
export default router;
