import { Router } from 'express';
import { authenticateToken } from '../middleware/auth.js';
const router = Router();
let steamcmdManager;
let logger;
export function setSteamCMDManager(manager, loggerInstance) {
    steamcmdManager = manager;
    logger = loggerInstance;
}
router.get('/status', authenticateToken, async (req, res) => {
    try {
        const status = await steamcmdManager.getStatus();
        res.json({
            success: true,
            data: status
        });
    }
    catch (error) {
        logger.error('获取SteamCMD状态失败:', error);
        res.status(500).json({
            success: false,
            message: '获取SteamCMD状态失败',
            error: error instanceof Error ? error.message : '未知错误'
        });
    }
});
router.post('/install', authenticateToken, async (req, res) => {
    try {
        const { installPath } = req.body;
        if (!installPath || typeof installPath !== 'string') {
            return res.status(400).json({
                success: false,
                message: '请提供有效的安装路径'
            });
        }
        res.writeHead(200, {
            'Content-Type': 'text/event-stream',
            'Cache-Control': 'no-cache',
            'Connection': 'keep-alive',
            'Access-Control-Allow-Origin': process.env.CORS_ORIGIN || '*',
            'Access-Control-Allow-Headers': 'Cache-Control'
        });
        const sendEvent = (event, data) => {
            res.write(`event: ${event}\n`);
            res.write(`data: ${JSON.stringify(data)}\n\n`);
        };
        try {
            await steamcmdManager.installOnline({
                installPath,
                onProgress: (progress) => {
                    sendEvent('progress', { progress });
                },
                onStatusChange: (status) => {
                    sendEvent('status', { status });
                }
            });
            sendEvent('complete', { success: true, message: 'SteamCMD安装完成' });
            res.end();
        }
        catch (error) {
            sendEvent('error', {
                success: false,
                message: 'SteamCMD安装失败',
                error: error instanceof Error ? error.message : '未知错误'
            });
            res.end();
        }
    }
    catch (error) {
        logger.error('SteamCMD安装请求处理失败:', error);
        if (!res.headersSent) {
            res.status(500).json({
                success: false,
                message: 'SteamCMD安装请求处理失败',
                error: error instanceof Error ? error.message : '未知错误'
            });
        }
    }
});
router.post('/manual-path', authenticateToken, async (req, res) => {
    try {
        const { installPath } = req.body;
        if (!installPath || typeof installPath !== 'string') {
            return res.status(400).json({
                success: false,
                message: '请提供有效的安装路径'
            });
        }
        const isInstalled = await steamcmdManager.setManualPath(installPath);
        res.json({
            success: true,
            data: {
                isInstalled,
                installPath,
                message: isInstalled ? 'SteamCMD路径设置成功' : 'SteamCMD路径已设置，但未找到可执行文件'
            }
        });
    }
    catch (error) {
        logger.error('设置SteamCMD手动路径失败:', error);
        res.status(500).json({
            success: false,
            message: '设置SteamCMD手动路径失败',
            error: error instanceof Error ? error.message : '未知错误'
        });
    }
});
router.post('/check-path', authenticateToken, async (req, res) => {
    try {
        const { installPath } = req.body;
        if (!installPath || typeof installPath !== 'string') {
            return res.status(400).json({
                success: false,
                message: '请提供有效的路径'
            });
        }
        const exists = await steamcmdManager.checkSteamCMDExists(installPath);
        res.json({
            success: true,
            data: {
                exists,
                path: installPath
            }
        });
    }
    catch (error) {
        logger.error('检查SteamCMD路径失败:', error);
        res.status(500).json({
            success: false,
            message: '检查SteamCMD路径失败',
            error: error instanceof Error ? error.message : '未知错误'
        });
    }
});
router.post('/refresh', authenticateToken, async (req, res) => {
    try {
        const status = await steamcmdManager.refreshStatus();
        res.json({
            success: true,
            data: status
        });
    }
    catch (error) {
        logger.error('刷新SteamCMD状态失败:', error);
        res.status(500).json({
            success: false,
            message: '刷新SteamCMD状态失败',
            error: error instanceof Error ? error.message : '未知错误'
        });
    }
});
router.get('/executable-path', authenticateToken, async (req, res) => {
    try {
        const executablePath = await steamcmdManager.getSteamCMDExecutablePath();
        res.json({
            success: true,
            data: {
                executablePath
            }
        });
    }
    catch (error) {
        logger.error('获取SteamCMD可执行文件路径失败:', error);
        res.status(500).json({
            success: false,
            message: '获取SteamCMD可执行文件路径失败',
            error: error instanceof Error ? error.message : '未知错误'
        });
    }
});
export default router;
