import { ChildProcess } from 'child_process';
export type LogLevel = 'info' | 'error' | 'success' | 'warn';
export type LogCallback = (message: string, type?: LogLevel) => void;
export interface DownloadProgress {
    loaded: number;
    total: number;
    percentage: number;
}
export type ProgressCallback = (progress: DownloadProgress) => void;
export type GameType = 'minecraft' | 'tmodloader' | 'factorio' | 'mrpack' | 'bedrock';
export interface CancellationToken {
    isCancelled: boolean;
    cancel(): void;
    onCancelled(callback: () => void): void;
    throwIfCancelled(): void;
}
export interface ActiveDeployment {
    id: string;
    game: GameType;
    targetDirectory: string;
    startTime: Date;
    tempDirectories: string[];
    processes: ChildProcess[];
    cancellationToken: CancellationToken;
    onProgress?: LogCallback;
}
export interface DeploymentManager {
    activeDeployments: Map<string, ActiveDeployment>;
    createDeployment(game: GameType, targetDirectory: string, onProgress?: LogCallback, deploymentId?: string): ActiveDeployment;
    cancelDeployment(deploymentId: string): Promise<boolean>;
    cancelAllDeployments(): Promise<number>;
    getActiveDeployments(): ActiveDeployment[];
    cleanupDeployment(deploymentId: string): Promise<void>;
}
export interface MinecraftServerInfo {
    name: string;
    displayName: string;
}
export interface MinecraftServerCategory {
    name: string;
    displayName: string;
    servers: MinecraftServerInfo[];
}
export interface MinecraftDownloadData {
    url: string;
    filename: string;
    size?: number;
}
export interface MinecraftDeployOptions {
    server: string;
    version: string;
    targetDirectory: string;
    deploymentId?: string;
    skipJavaCheck?: boolean;
    skipServerRun?: boolean;
    onProgress?: ProgressCallback;
    onLog?: LogCallback;
}
export interface TModLoaderOptions {
    downloadDir?: string;
    extractDir?: string;
    deleteAfterExtract?: boolean;
    clearExtractDir?: boolean;
    createVersionDir?: boolean;
}
export interface TModLoaderInfo {
    version: string;
    downloadUrl: string;
}
export interface TModLoaderDeployOptions {
    targetDirectory: string;
    options?: TModLoaderOptions;
    deploymentId?: string;
    onProgress?: LogCallback;
}
export interface FactorioDeployOptions {
    targetDirectory: string;
    tempDir?: string;
    deploymentId?: string;
    onProgress?: LogCallback;
}
export interface BedrockDownloadLink {
    downloadType: string;
    downloadUrl: string;
}
export interface BedrockVersionInfo {
    links: BedrockDownloadLink[];
}
export interface BedrockDeployOptions {
    targetDirectory: string;
    platform?: 'windows' | 'linux';
    versionType?: 'stable' | 'preview';
    deploymentId?: string;
    onProgress?: LogCallback;
}
export interface MrpackSearchOptions {
    query?: string;
    categories?: string[];
    versions?: string[];
    license?: string;
    limit?: number;
    offset?: number;
    index?: string;
}
export interface MrpackProject {
    project_id: string;
    slug: string;
    author: string;
    title: string;
    description: string;
    categories: string[];
    versions: string[];
    downloads: number;
    icon_url?: string;
    latest_version?: string;
}
export interface MrpackSearchResponse {
    hits: MrpackProject[];
    offset: number;
    limit: number;
    total_hits: number;
}
export interface MrpackDeployOptions {
    projectId?: string;
    versionId?: string;
    mrpackUrl?: string;
    targetDirectory: string;
    minecraftVersion?: string;
    loaderType?: 'forge' | 'fabric' | 'quilt';
    skipJavaCheck?: boolean;
    tempDir?: string;
    deploymentId?: string;
    options?: any;
    onProgress?: LogCallback;
}
export interface MrpackIndex {
    formatVersion: number;
    game: string;
    versionId: string;
    name: string;
    summary?: string;
    files: MrpackFile[];
    dependencies: Record<string, string>;
}
export interface MrpackFile {
    path: string;
    hashes: {
        sha1: string;
        sha512?: string;
    };
    env?: {
        client?: string;
        server?: string;
    };
    downloads: string[];
    fileSize: number;
}
export interface DeploymentResult {
    success: boolean;
    message: string;
    targetDirectory?: string;
    details?: any;
    deploymentId?: string;
    data?: any;
}
export declare function getActiveDeployments(): ActiveDeployment[];
export declare function cancelDeployment(deploymentId: string): Promise<boolean>;
export declare function cancelAllDeployments(): Promise<number>;
export declare function cleanupDeployment(deploymentId: string): Promise<void>;
export declare function createTempDirectory(deployment: ActiveDeployment, prefix?: string): Promise<string>;
export declare function getMinecraftServerCategories(): Promise<MinecraftServerCategory[]>;
export declare function getMinecraftVersions(server: string): Promise<string[]>;
export declare function getMinecraftDownloadInfo(server: string, version: string): Promise<MinecraftDownloadData>;
export declare function validateJavaEnvironment(): Promise<boolean>;
export declare function downloadFile(url: string, filePath: string, onProgress?: ProgressCallback, onLog?: LogCallback): Promise<void>;
export declare function downloadFileWithCancellation(url: string, filePath: string, deployment: ActiveDeployment, onProgress?: ProgressCallback, onLog?: LogCallback): Promise<void>;
export declare function deployMinecraftServer(options: MinecraftDeployOptions): Promise<DeploymentResult>;
export declare function getTModLoaderInfo(): Promise<TModLoaderInfo>;
export declare function deployTModLoaderServer(options: TModLoaderDeployOptions): Promise<DeploymentResult>;
export declare function deployFactorioServer(options: FactorioDeployOptions): Promise<DeploymentResult>;
export declare function searchMrpackModpacks(options?: MrpackSearchOptions): Promise<MrpackSearchResponse>;
export declare function getMrpackProjectVersions(projectId: string): Promise<any[]>;
export declare function downloadAndParseMrpack(mrpackUrl: string): Promise<MrpackIndex>;
export declare function deployMrpackServer(options: MrpackDeployOptions): Promise<DeploymentResult>;
export declare function extractZipFile(zipPath: string, extractPath: string): Promise<void>;
export declare function extractZipFileWithCancellation(zipPath: string, extractPath: string, deployment: ActiveDeployment): Promise<void>;
export declare function extractTarXzFile(filePath: string, extractPath: string): Promise<void>;
export declare function extractTarXzFileWithCancellation(filePath: string, extractPath: string, deployment: ActiveDeployment): Promise<void>;
export interface UnifiedDeployOptions {
    game: GameType;
    targetDirectory: string;
    platform?: 'windows' | 'linux';
    server?: string;
    version?: string;
    skipJavaCheck?: boolean;
    skipServerRun?: boolean;
    tmodOptions?: TModLoaderOptions;
    tempDir?: string;
    mrpackUrl?: string;
    minecraftVersion?: string;
    onProgress?: LogCallback;
}
export declare function deployGameServer(options: UnifiedDeployOptions): Promise<DeploymentResult>;
export declare function getBedrockDownloadLinks(): Promise<BedrockVersionInfo>;
export declare function deployBedrockServer(options: BedrockDeployOptions): Promise<DeploymentResult>;
export default deployGameServer;
