import { Router } from 'express';
import axios from 'axios';
import { authenticateToken } from '../middleware/auth.js';
import path from 'path';
import fs from 'fs-extra';
import AdmZip from 'adm-zip';
import { pipeline } from 'stream/promises';
import { createWriteStream } from 'fs';
const router = Router();
const MSL_BUILD_SERVER = 'https://download.mc.xiaozhuhouses.asia:4433';
router.get('/:type/cores', authenticateToken, async (req, res) => {
    try {
        const { type } = req.params;
        const response = await axios.get(`${MSL_BUILD_SERVER}/api/${type}/cores`, {
            timeout: 30000
        });
        res.json(response.data);
    }
    catch (error) {
        console.error('获取云构建核心列表失败:', error.message);
        res.status(error.response?.status || 500).json({
            success: false,
            message: error.response?.data?.message || '无法连接到MSL云构建服务',
            error: error.message
        });
    }
});
router.post('/build', authenticateToken, async (req, res) => {
    try {
        const { coreName, version, type = 'msl_Official' } = req.body;
        if (!coreName || !version) {
            return res.status(400).json({
                success: false,
                message: '缺少必要参数：coreName 和 version'
            });
        }
        const response = await axios.post(`${MSL_BUILD_SERVER}/api/build`, {
            coreName,
            version,
            type
        }, {
            timeout: 60000
        });
        res.json(response.data);
    }
    catch (error) {
        console.error('创建云构建任务失败:', error.message);
        if (error.response?.status === 429) {
            return res.status(429).json(error.response.data);
        }
        res.status(error.response?.status || 500).json({
            success: false,
            message: error.response?.data?.message || '创建构建任务失败',
            error: error.message
        });
    }
});
router.get('/build/:taskId', authenticateToken, async (req, res) => {
    try {
        const { taskId } = req.params;
        const response = await axios.get(`${MSL_BUILD_SERVER}/api/build/${taskId}`, {
            timeout: 30000
        });
        res.json(response.data);
    }
    catch (error) {
        console.error('查询构建任务状态失败:', error.message);
        res.status(error.response?.status || 500).json({
            success: false,
            message: error.response?.data?.message || '查询构建任务状态失败',
            error: error.message
        });
    }
});
router.post('/download', authenticateToken, async (req, res) => {
    try {
        const { downloadUrl, fileName, coreName, version, targetPath } = req.body;
        if (!targetPath) {
            return res.status(400).json({
                success: false,
                message: '缺少必要参数：targetPath'
            });
        }
        let finalDownloadUrl = downloadUrl;
        if (!downloadUrl && fileName) {
            const linkResponse = await axios.post(`${MSL_BUILD_SERVER}/api/download`, {
                fileName,
                coreName,
                version
            }, {
                timeout: 30000
            });
            if (!linkResponse.data.success || !linkResponse.data.data.downloadUrl) {
                throw new Error('生成下载链接失败');
            }
            finalDownloadUrl = linkResponse.data.data.downloadUrl;
        }
        if (!finalDownloadUrl) {
            return res.status(400).json({
                success: false,
                message: '缺少必要参数：downloadUrl 或 fileName'
            });
        }
        const downloadFileName = fileName || `${coreName}-${version}.zip`;
        await fs.ensureDir(targetPath);
        const tempDir = path.join(process.cwd(), 'server', 'temp');
        await fs.ensureDir(tempDir);
        const tempFilePath = path.join(tempDir, downloadFileName);
        const downloadResponse = await axios({
            method: 'get',
            url: `${MSL_BUILD_SERVER}${finalDownloadUrl}`,
            responseType: 'stream',
            timeout: 300000
        });
        await pipeline(downloadResponse.data, createWriteStream(tempFilePath));
        const zip = new AdmZip(tempFilePath);
        zip.extractAllTo(targetPath, true);
        await fs.remove(tempFilePath);
        const files = await fs.readdir(targetPath);
        let startCommand = '';
        const isWindows = process.platform === 'win32';
        if (isWindows) {
            const runBat = files.find(file => file.toLowerCase() === 'run.bat');
            const startBat = files.find(file => file.toLowerCase() === 'start.bat');
            if (runBat) {
                startCommand = `.\\${runBat}`;
            }
            else if (startBat) {
                startCommand = `.\\${startBat}`;
            }
        }
        else {
            const runSh = files.find(file => file.toLowerCase() === 'run.sh');
            const startSh = files.find(file => file.toLowerCase() === 'start.sh');
            if (runSh) {
                startCommand = `bash ${runSh}`;
            }
            else if (startSh) {
                startCommand = `bash ${startSh}`;
            }
        }
        res.json({
            success: true,
            message: '下载并解压成功',
            data: {
                targetPath,
                startCommand,
                files: files.length
            }
        });
    }
    catch (error) {
        console.error('下载并解压失败:', error.message);
        res.status(error.response?.status || 500).json({
            success: false,
            message: error.response?.data?.message || '下载并解压失败',
            error: error.message
        });
    }
});
router.get('/download/:linkId', authenticateToken, async (req, res) => {
    try {
        const { linkId } = req.params;
        const response = await axios.get(`${MSL_BUILD_SERVER}/api/download/${linkId}`, {
            responseType: 'stream',
            timeout: 300000
        });
        res.setHeader('Content-Type', response.headers['content-type'] || 'application/zip');
        res.setHeader('Content-Disposition', response.headers['content-disposition'] || 'attachment');
        if (response.headers['content-length']) {
            res.setHeader('Content-Length', response.headers['content-length']);
        }
        response.data.pipe(res);
    }
    catch (error) {
        console.error('下载文件失败:', error.message);
        if (!res.headersSent) {
            res.status(error.response?.status || 500).json({
                success: false,
                message: error.response?.data?.message || '下载文件失败',
                error: error.message
            });
        }
    }
});
router.get('/stats', authenticateToken, async (req, res) => {
    try {
        const response = await axios.get(`${MSL_BUILD_SERVER}/api/stats`, {
            timeout: 30000
        });
        res.json(response.data);
    }
    catch (error) {
        console.error('获取云构建统计数据失败:', error.message);
        res.status(error.response?.status || 500).json({
            success: false,
            message: error.response?.data?.message || '获取统计数据失败',
            error: error.message
        });
    }
});
router.get('/cache/modrinth', authenticateToken, async (req, res) => {
    try {
        const response = await axios.get(`${MSL_BUILD_SERVER}/api/cache/modrinth`, {
            timeout: 30000
        });
        res.json(response.data);
    }
    catch (error) {
        console.error('获取 Modrinth 缓存列表失败:', error.message);
        res.status(error.response?.status || 500).json({
            success: false,
            message: error.response?.data?.message || '获取 Modrinth 缓存列表失败',
            error: error.message
        });
    }
});
export default router;
