import { Router } from 'express';
import { authenticateToken } from '../middleware/auth.js';
import net from 'net';
import http from 'http';
import https from 'https';
import { URL } from 'url';
const router = Router();
const networkCheckItems = [
    { id: 'baidu', name: '互联网连接状态', url: 'www.baidu.com', status: 'pending' },
    { id: 'steamworks-api', name: 'Steamworks API(全球)', url: 'api.steampowered.com', status: 'pending' },
    { id: 'steamworks-partner', name: 'Steamworks API（合作/私有）', url: 'partner.steam-api.com', status: 'pending' },
    { id: 'modrinth-api', name: 'Modrinth API', url: 'api.modrinth.com', status: 'pending' },
    { id: 'modrinth-cdn', name: 'Modrinth CDN', url: 'cdn.modrinth.com', status: 'pending' },
    { id: 'mojang-session', name: 'Mojang 会话服务器', url: 'sessionserver.mojang.com', status: 'pending' },
    { id: 'msl-api', name: 'MSL API', url: 'https://api.mslmc.cn/v3', status: 'pending' },
    { id: 'gsm-deploy', name: 'GSManager在线部署服务', url: 'http://api.gsm.xiaozhuhouses.asia:10002/', status: 'pending' },
    { id: 'gsm-mirror', name: 'GSManager镜像服务器', url: 'http://download.xiaozhuhouses.asia:8082/', status: 'pending' },
    { id: 'gsm-cloud-build', name: 'GSManager 云构建服务', url: 'http://api.mc.xiaozhuhouses.asia:10003', status: 'pending' },
    { id: 'gsm-cloud-cache', name: 'GSManager 云构建缓存节点', url: 'https://download.mc.xiaozhuhouses.asia:4433', status: 'pending' }
];
function tcpPing(host, port, timeout = 10000) {
    return new Promise((resolve) => {
        const startTime = Date.now();
        const socket = new net.Socket();
        const timeoutId = setTimeout(() => {
            socket.destroy();
            resolve({ success: false, error: `连接超时 (${timeout}ms)` });
        }, timeout);
        socket.setTimeout(timeout);
        socket.connect(port, host, () => {
            const responseTime = Date.now() - startTime;
            clearTimeout(timeoutId);
            socket.destroy();
            resolve({ success: true, responseTime });
        });
        socket.on('error', (err) => {
            clearTimeout(timeoutId);
            socket.destroy();
            resolve({ success: false, error: err.message });
        });
        socket.on('timeout', () => {
            clearTimeout(timeoutId);
            socket.destroy();
            resolve({ success: false, error: `连接超时 (${timeout}ms)` });
        });
    });
}
function httpPing(url, timeout = 10000) {
    return new Promise((resolve) => {
        const startTime = Date.now();
        try {
            let parsedUrl;
            if (!url.startsWith('http://') && !url.startsWith('https://')) {
                try {
                    parsedUrl = new URL(`https://${url}`);
                }
                catch {
                    parsedUrl = new URL(`http://${url}`);
                }
            }
            else {
                parsedUrl = new URL(url);
            }
            const protocol = parsedUrl.protocol === 'https:' ? https : http;
            const req = protocol.get(parsedUrl.toString(), {
                timeout,
                headers: {
                    'User-Agent': 'GSManager-NetworkCheck/1.0'
                }
            }, (res) => {
                const responseTime = Date.now() - startTime;
                res.resume();
                resolve({ success: true, responseTime });
            });
            req.on('error', (err) => {
                resolve({ success: false, error: err.message });
            });
            req.on('timeout', () => {
                req.destroy();
                resolve({ success: false, error: `连接超时 (${timeout}ms)` });
            });
        }
        catch (error) {
            resolve({ success: false, error: error.message });
        }
    });
}
async function checkUrl(url, timeout = 10000) {
    try {
        if (url.startsWith('http://') || url.startsWith('https://')) {
            return await httpPing(url, timeout);
        }
        const httpResult = await httpPing(url, timeout);
        if (httpResult.success) {
            return httpResult;
        }
        const hostname = url.replace(/^(https?:\/\/)/, '');
        return await tcpPing(hostname, 80, timeout);
    }
    catch (error) {
        return { success: false, error: error.message };
    }
}
router.get('/check-all', authenticateToken, async (req, res) => {
    try {
        const results = await Promise.all(networkCheckItems.map(async (item) => {
            const result = await checkUrl(item.url, 10000);
            return {
                id: item.id,
                name: item.name,
                url: item.url,
                status: result.success ? 'success' : 'failed',
                responseTime: result.responseTime,
                error: result.error
            };
        }));
        res.json({
            success: true,
            data: {
                results,
                timestamp: new Date().toISOString()
            }
        });
    }
    catch (error) {
        console.error('网络检测失败:', error);
        res.status(500).json({
            success: false,
            message: '网络检测失败',
            error: error.message
        });
    }
});
router.post('/check-single', authenticateToken, async (req, res) => {
    try {
        const { url } = req.body;
        if (!url) {
            return res.status(400).json({
                success: false,
                message: '缺少URL参数'
            });
        }
        const result = await checkUrl(url, 10000);
        res.json({
            success: true,
            data: {
                url,
                status: result.success ? 'success' : 'failed',
                responseTime: result.responseTime,
                error: result.error,
                timestamp: new Date().toISOString()
            }
        });
    }
    catch (error) {
        console.error('单项网络检测失败:', error);
        res.status(500).json({
            success: false,
            message: '网络检测失败',
            error: error.message
        });
    }
});
export default router;
