import { Router } from 'express';
import { authenticateToken } from '../middleware/auth.js';
import logger from '../utils/logger.js';
const router = Router();
let configManager;
export function setSecurityConfigManager(manager) {
    configManager = manager;
}
router.get('/config', authenticateToken, async (req, res) => {
    try {
        const securityConfig = configManager.getSecurityConfig();
        const jwtConfig = configManager.getJWTConfig();
        res.json({
            success: true,
            data: {
                tokenResetRule: securityConfig.tokenResetRule,
                tokenExpireHours: securityConfig.tokenExpireHours,
                currentExpiresIn: jwtConfig.expiresIn
            }
        });
    }
    catch (error) {
        logger.error('获取安全配置失败:', error);
        res.status(500).json({
            success: false,
            message: '获取安全配置失败'
        });
    }
});
router.post('/config', authenticateToken, async (req, res) => {
    try {
        const { tokenResetRule, tokenExpireHours } = req.body;
        if (!tokenResetRule || !['startup', 'expire'].includes(tokenResetRule)) {
            return res.status(400).json({
                success: false,
                message: '无效的令牌重置规则'
            });
        }
        if (tokenExpireHours !== null && (typeof tokenExpireHours !== 'number' || tokenExpireHours <= 0)) {
            return res.status(400).json({
                success: false,
                message: '令牌到期时间必须为正数或null'
            });
        }
        await configManager.updateSecurityConfig({
            tokenResetRule,
            tokenExpireHours
        });
        if (tokenExpireHours !== null) {
            await configManager.updateJWTConfig({
                expiresIn: `${tokenExpireHours}h`
            });
        }
        else {
            await configManager.updateJWTConfig({
                expiresIn: '876000h'
            });
        }
        logger.info(`安全配置已更新: 重置规则=${tokenResetRule}, 到期时间=${tokenExpireHours}小时`);
        res.json({
            success: true,
            message: '安全配置已更新'
        });
    }
    catch (error) {
        logger.error('更新安全配置失败:', error);
        res.status(500).json({
            success: false,
            message: '更新安全配置失败'
        });
    }
});
router.post('/reset-token', authenticateToken, async (req, res) => {
    try {
        const securityConfig = configManager.getSecurityConfig();
        if (securityConfig.tokenResetRule === 'startup') {
            await configManager.regenerateJWTSecret();
            logger.info('JWT密钥已重新生成（启动时重置规则）');
            res.json({
                success: true,
                message: 'JWT密钥已重新生成，所有现有令牌将失效'
            });
        }
        else {
            res.json({
                success: true,
                message: '当前设置为过期自动重置，无需立即操作'
            });
        }
    }
    catch (error) {
        logger.error('重置令牌失败:', error);
        res.status(500).json({
            success: false,
            message: '重置令牌失败'
        });
    }
});
export default router;
