import { Router } from 'express';
import { authenticateToken } from '../middleware/auth.js';
import logger from '../utils/logger.js';
const router = Router();
let configManager;
export function setConfigManager(manager) {
    configManager = manager;
}
router.post('/game-path', authenticateToken, async (req, res) => {
    try {
        if (!configManager) {
            return res.status(500).json({
                success: false,
                error: 'ConfigManager未初始化'
            });
        }
        const { defaultGamePath } = req.body;
        if (!defaultGamePath || typeof defaultGamePath !== 'string') {
            return res.status(400).json({
                success: false,
                error: '参数错误',
                message: 'defaultGamePath必须是非空字符串'
            });
        }
        const trimmedPath = defaultGamePath.trim();
        if (!trimmedPath) {
            return res.status(400).json({
                success: false,
                error: '参数错误',
                message: '路径不能为空'
            });
        }
        await configManager.updateGameConfig({ defaultInstallPath: trimmedPath });
        logger.info(`游戏默认安装路径已设置: ${trimmedPath}`);
        res.json({
            success: true,
            message: '游戏默认安装路径设置成功',
            data: {
                defaultInstallPath: trimmedPath
            }
        });
    }
    catch (error) {
        logger.error('设置游戏默认安装路径失败:', error);
        res.status(500).json({
            success: false,
            error: '服务器内部错误',
            message: '设置游戏默认安装路径失败'
        });
    }
});
router.get('/game-path', authenticateToken, (req, res) => {
    try {
        if (!configManager) {
            return res.status(500).json({
                success: false,
                error: 'ConfigManager未初始化'
            });
        }
        const gameConfig = configManager.getGameConfig();
        res.json({
            success: true,
            data: {
                defaultInstallPath: gameConfig.defaultInstallPath || ''
            }
        });
    }
    catch (error) {
        logger.error('获取游戏默认安装路径失败:', error);
        res.status(500).json({
            success: false,
            error: '服务器内部错误',
            message: '获取游戏默认安装路径失败'
        });
    }
});
export function setupSettingsRoutes(manager) {
    setConfigManager(manager);
    return router;
}
export default router;
