import { Router } from 'express';
import { authenticateToken } from '../middleware/auth.js';
import { taskManager } from '../modules/task/taskManager.js';
import logger from '../utils/logger.js';
const router = Router();
router.get('/', authenticateToken, async (req, res) => {
    try {
        const tasks = taskManager.getAllTasks();
        res.json({
            success: true,
            data: tasks
        });
    }
    catch (error) {
        logger.error('获取任务列表失败:', error);
        res.status(500).json({
            success: false,
            message: '获取任务列表失败',
            error: error.message
        });
    }
});
router.get('/active', authenticateToken, async (req, res) => {
    try {
        const tasks = taskManager.getActiveTasks();
        res.json({
            success: true,
            data: tasks
        });
    }
    catch (error) {
        logger.error('获取活跃任务失败:', error);
        res.status(500).json({
            success: false,
            message: '获取活跃任务失败',
            error: error.message
        });
    }
});
router.get('/:taskId', authenticateToken, async (req, res) => {
    try {
        const { taskId } = req.params;
        const task = taskManager.getTask(taskId);
        if (!task) {
            return res.status(404).json({
                success: false,
                message: '任务不存在'
            });
        }
        res.json({
            success: true,
            data: task
        });
    }
    catch (error) {
        logger.error('获取任务失败:', error);
        res.status(500).json({
            success: false,
            message: '获取任务失败',
            error: error.message
        });
    }
});
router.delete('/:taskId', authenticateToken, async (req, res) => {
    try {
        const { taskId } = req.params;
        const task = taskManager.getTask(taskId);
        if (!task) {
            return res.status(404).json({
                success: false,
                message: '任务不存在'
            });
        }
        if (task.status === 'pending' || task.status === 'running') {
            return res.status(400).json({
                success: false,
                message: '无法删除正在进行的任务'
            });
        }
        taskManager.deleteTask(taskId);
        res.json({
            success: true,
            message: '任务删除成功'
        });
    }
    catch (error) {
        logger.error('删除任务失败:', error);
        res.status(500).json({
            success: false,
            message: '删除任务失败',
            error: error.message
        });
    }
});
export default router;
