import { Router } from 'express';
import logger from '../utils/logger.js';
const router = Router();
let terminalManager;
export function setTerminalManager(manager) {
    terminalManager = manager;
}
router.get('/stats', (req, res) => {
    try {
        if (!terminalManager) {
            return res.status(500).json({ error: '终端管理器未初始化' });
        }
        const stats = terminalManager.getSessionStats();
        res.json({
            success: true,
            data: stats
        });
    }
    catch (error) {
        logger.error('获取终端统计失败:', error);
        res.status(500).json({
            success: false,
            error: error instanceof Error ? error.message : '获取统计失败'
        });
    }
});
router.get('/sessions', (req, res) => {
    try {
        if (!terminalManager) {
            return res.status(500).json({ error: '终端管理器未初始化' });
        }
        const stats = terminalManager.getSessionStats();
        const savedSessions = terminalManager.getSavedSessions();
        res.json({
            success: true,
            data: {
                activeSessions: stats.sessions,
                savedSessions: savedSessions,
                total: stats.total
            }
        });
    }
    catch (error) {
        logger.error('获取终端会话列表失败:', error);
        res.status(500).json({
            success: false,
            error: error instanceof Error ? error.message : '获取会话列表失败'
        });
    }
});
router.get('/active-processes', async (req, res) => {
    try {
        if (!terminalManager) {
            return res.status(500).json({ error: '终端管理器未初始化' });
        }
        const activeProcesses = await terminalManager.getActiveTerminalProcesses();
        res.json({
            success: true,
            data: activeProcesses
        });
    }
    catch (error) {
        logger.error('获取活跃终端进程失败:', error);
        res.status(500).json({
            success: false,
            error: error instanceof Error ? error.message : '获取活跃终端进程失败'
        });
    }
});
router.put('/sessions/:sessionId/name', async (req, res) => {
    try {
        if (!terminalManager) {
            return res.status(500).json({ error: '终端管理器未初始化' });
        }
        const { sessionId } = req.params;
        const { name } = req.body;
        if (!name || typeof name !== 'string' || name.trim().length === 0) {
            return res.status(400).json({
                success: false,
                error: '会话名称不能为空'
            });
        }
        const success = await terminalManager.updateSessionName(sessionId, name.trim());
        if (success) {
            res.json({
                success: true,
                message: '会话名称更新成功'
            });
        }
        else {
            res.status(404).json({
                success: false,
                error: '会话不存在'
            });
        }
    }
    catch (error) {
        logger.error('更新会话名称失败:', error);
        res.status(500).json({
            success: false,
            error: error instanceof Error ? error.message : '更新会话名称失败'
        });
    }
});
router.post('/validate-config', (req, res) => {
    try {
        const { workingDirectory, shell } = req.body;
        const errors = [];
        if (!workingDirectory) {
            errors.push('工作目录不能为空');
        }
        if (shell && typeof shell !== 'string') {
            errors.push('Shell配置格式错误');
        }
        if (errors.length > 0) {
            return res.status(400).json({
                success: false,
                errors
            });
        }
        res.json({
            success: true,
            message: '配置验证通过'
        });
    }
    catch (error) {
        logger.error('验证终端配置失败:', error);
        res.status(500).json({
            success: false,
            error: error instanceof Error ? error.message : '配置验证失败'
        });
    }
});
router.get('/default-shell', (req, res) => {
    try {
        const platform = process.platform;
        let defaultShell;
        let availableShells;
        if (platform === 'win32') {
            defaultShell = 'powershell.exe';
            availableShells = [
                'powershell.exe',
                'cmd.exe',
                'pwsh.exe'
            ];
        }
        else {
            defaultShell = process.env.SHELL || '/bin/bash';
            availableShells = [
                '/bin/bash',
                '/bin/sh',
                '/bin/zsh',
                '/bin/fish'
            ];
        }
        res.json({
            success: true,
            data: {
                platform,
                defaultShell,
                availableShells,
                currentShell: process.env.SHELL || defaultShell
            }
        });
    }
    catch (error) {
        logger.error('获取默认Shell失败:', error);
        res.status(500).json({
            success: false,
            error: error instanceof Error ? error.message : '获取默认Shell失败'
        });
    }
});
router.get('/themes', (req, res) => {
    try {
        const themes = [
            {
                name: 'default',
                displayName: '默认',
                colors: {
                    background: '#000000',
                    foreground: '#ffffff',
                    cursor: '#ffffff',
                    selection: '#ffffff40'
                }
            },
            {
                name: 'dark',
                displayName: '深色',
                colors: {
                    background: '#1e1e1e',
                    foreground: '#d4d4d4',
                    cursor: '#d4d4d4',
                    selection: '#264f78'
                }
            },
            {
                name: 'light',
                displayName: '浅色',
                colors: {
                    background: '#ffffff',
                    foreground: '#000000',
                    cursor: '#000000',
                    selection: '#0078d4'
                }
            },
            {
                name: 'monokai',
                displayName: 'Monokai',
                colors: {
                    background: '#272822',
                    foreground: '#f8f8f2',
                    cursor: '#f8f8f0',
                    selection: '#49483e'
                }
            },
            {
                name: 'solarized-dark',
                displayName: 'Solarized Dark',
                colors: {
                    background: '#002b36',
                    foreground: '#839496',
                    cursor: '#93a1a1',
                    selection: '#073642'
                }
            }
        ];
        res.json({
            success: true,
            data: themes
        });
    }
    catch (error) {
        logger.error('获取终端主题失败:', error);
        res.status(500).json({
            success: false,
            error: error instanceof Error ? error.message : '获取主题失败'
        });
    }
});
router.get('/fonts', (req, res) => {
    try {
        const fonts = [
            {
                family: 'Consolas',
                displayName: 'Consolas',
                monospace: true
            },
            {
                family: 'Monaco',
                displayName: 'Monaco',
                monospace: true
            },
            {
                family: 'Menlo',
                displayName: 'Menlo',
                monospace: true
            },
            {
                family: 'Courier New',
                displayName: 'Courier New',
                monospace: true
            },
            {
                family: 'monospace',
                displayName: '系统等宽字体',
                monospace: true
            },
            {
                family: 'Fira Code',
                displayName: 'Fira Code',
                monospace: true,
                ligatures: true
            },
            {
                family: 'Source Code Pro',
                displayName: 'Source Code Pro',
                monospace: true
            },
            {
                family: 'JetBrains Mono',
                displayName: 'JetBrains Mono',
                monospace: true,
                ligatures: true
            }
        ];
        const sizes = [8, 9, 10, 11, 12, 13, 14, 15, 16, 18, 20, 22, 24, 26, 28, 30];
        res.json({
            success: true,
            data: {
                fonts,
                sizes,
                defaultFont: 'Consolas',
                defaultSize: 14
            }
        });
    }
    catch (error) {
        logger.error('获取终端字体失败:', error);
        res.status(500).json({
            success: false,
            error: error instanceof Error ? error.message : '获取字体失败'
        });
    }
});
router.post('/test-connection', (req, res) => {
    try {
        const { workingDirectory } = req.body;
        res.json({
            success: true,
            message: '终端连接测试成功',
            data: {
                workingDirectory: workingDirectory || process.cwd(),
                platform: process.platform,
                shell: process.env.SHELL || (process.platform === 'win32' ? 'powershell.exe' : '/bin/bash')
            }
        });
    }
    catch (error) {
        logger.error('测试终端连接失败:', error);
        res.status(500).json({
            success: false,
            error: error instanceof Error ? error.message : '连接测试失败'
        });
    }
});
export function setupTerminalRoutes(manager) {
    setTerminalManager(manager);
    return router;
}
export default router;
